#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Anthro Simple Player v2.1

A simple, functional, and optimized music player for MP3, WAV, and FLAC files.
This version features a button to physically shuffle the playlist order.

Features:
- Supports .mp3, .wav, and .flac formats.
- Drag-and-drop to add songs (via tkinterdnd2).
  - File scanning is threaded to keep the UI responsive.
- A "Shuffle" button that randomizes the current playlist's order.
- Playback progress bar with seeking (click/drag to jump).
- Current time and total duration display.
- Volume control.
- Keyboard shortcuts (Space for Play/Pause, Arrow keys for Next/Prev).
- Save/Open playlists in a simple JSON format (.ahpl).
- Auto-saves the last session and restores it on launch.

Dependencies:
    pip install pygame
    # Optional for OS drag-and-drop:
    pip install tkinterdnd2

Run:
    python anthro_simple_player_v2.1.py
"""
import json
import random
import threading
from pathlib import Path
from tkinter import filedialog, messagebox, ttk
import tkinter as tk

# Try to import tkinterdnd2 for drag-and-drop
try:
    from tkinterdnd2 import DND_FILES, TkinterDnD
    DND_AVAILABLE = True
except ImportError:
    DND_AVAILABLE = False

# Try to import pygame for audio playback
try:
    import pygame
    PYGAME_AVAILABLE = True
except ImportError:
    PYGAME_AVAILABLE = False

# --- Constants ---
APP_NAME = "Anthro Simple Player"
APP_VERSION = "2.1"
APP_ID = "anthro_simple_player"
SUPPORTED_EXTS = {".mp3", ".wav", ".flac"}
PLAYLIST_EXT = ".ahpl"

CONFIG_DIR = Path.home() / f".{APP_ID}"
AUTO_PATH = CONFIG_DIR / f"last_playlist{PLAYLIST_EXT}"


# --- Utility Functions (unchanged) ---
def ensure_config_dir():
    try:
        CONFIG_DIR.mkdir(parents=True, exist_ok=True)
    except OSError:
        pass

def is_audio_file(path: Path) -> bool:
    return path.is_file() and path.suffix.lower() in SUPPORTED_EXTS

def find_audio_files_in(path: Path):
    path = path.resolve()
    if path.is_file() and is_audio_file(path):
        yield path
    elif path.is_dir():
        for item in path.rglob("*"):
            if is_audio_file(item):
                yield item

def parse_dnd_event_data(data: str) -> list[str]:
    if not data:
        return []
    paths = []
    in_brace = False
    current_path = ""
    for char in data:
        if char == '{' and not in_brace:
            in_brace = True
        elif char == '}' and in_brace:
            in_brace = False
            if current_path:
                paths.append(current_path)
                current_path = ""
        elif char == ' ' and not in_brace:
            if current_path:
                paths.append(current_path)
                current_path = ""
        else:
            current_path += char
    if current_path:
        paths.append(current_path)
    return [p.strip() for p in paths if p.strip()]


# --- Data Model ---
class Playlist:
    """Manages the list of tracks and current position."""
    def __init__(self):
        self.tracks: list[Path] = []
        self.current_index: int = -1
        self.volume: float = 0.8
        # --- REMOVED --- shuffle-related attributes are gone

    def add_paths(self, paths: list[Path]) -> int:
        existing = set(self.tracks)
        added_count = 0
        for p in paths:
            if p not in existing:
                self.tracks.append(p)
                added_count += 1
        return added_count

    def clear(self):
        self.tracks.clear()
        self.current_index = -1

    # --- ADDED --- New method to shuffle the playlist in-place.
    def shuffle_tracks(self):
        """
        Randomizes the order of tracks in the playlist.
        If a track is currently selected, it is moved to the top of the new list.
        """
        if len(self.tracks) < 2:
            return  # Not enough tracks to shuffle

        current_track = None
        if 0 <= self.current_index < len(self.tracks):
            current_track = self.tracks.pop(self.current_index)

        random.shuffle(self.tracks)

        if current_track:
            self.tracks.insert(0, current_track)
            self.current_index = 0
        else:
            # If nothing was playing, the index remains -1 until user action
            pass

    # --- MODIFIED --- Simplified next/prev logic
    def get_next_index(self) -> int:
        if not self.tracks:
            return -1
        return (self.current_index + 1) % len(self.tracks)

    def get_prev_index(self) -> int:
        if not self.tracks:
            return -1
        return (self.current_index - 1 + len(self.tracks)) % len(self.tracks)

    # --- MODIFIED --- to_dict and from_dict no longer handle shuffle state
    def to_dict(self) -> dict:
        return {
            "version": 2.1,
            "tracks": [str(p) for p in self.tracks],
            "current_index": self.current_index,
            "volume": self.volume,
        }

    def from_dict(self, data: dict):
        self.tracks = [Path(p) for p in data.get("tracks", []) if Path(p).exists()]
        self.current_index = int(data.get("current_index", -1))
        self.volume = float(data.get("volume", 0.8))


# --- Main Application ---
class PlayerApp:
    def __init__(self, root):
        self.root = root
        self.root.title(APP_NAME)
        self.root.geometry("750x550")
        self.root.minsize(500, 400)

        self.playlist = Playlist()
        self.is_playing = False
        self.is_paused = False
        self.is_seeking = False
        self.current_track_duration = 0
        self.playlist_path: Path | None = None

        self._init_pygame()
        self._setup_style()
        self._create_widgets()
        self._configure_bindings()
        
        ensure_config_dir()
        self._load_session()

        self.root.protocol("WM_DELETE_WINDOW", self.on_close)
        self._update_playback_progress()

    def _init_pygame(self):
        if not PYGAME_AVAILABLE:
            self.root.after(100, lambda: messagebox.showwarning(
                "Missing Dependency", "The 'pygame' package is required for audio playback."
            ))
            return
        try:
            pygame.init()
            pygame.mixer.init(frequency=44100, size=-16, channels=2, buffer=1024)
        except Exception as e:
            messagebox.showerror("Audio Init Failed", f"pygame.mixer failed to initialize:\n{e}")

    def _setup_style(self):
        style = ttk.Style(self.root)
        style.theme_use('clam')
        style.configure("TButton", padding=6, relief="flat", background="#ccc")

    def _create_widgets(self):
        self._create_menu()
        main_frame = ttk.Frame(self.root, padding=10)
        main_frame.pack(fill="both", expand=True)
        main_frame.rowconfigure(2, weight=1)
        main_frame.columnconfigure(0, weight=1)
        self._create_controls(main_frame)
        self._create_progress_display(main_frame)
        self._create_playlist_view(main_frame)
        self._create_status_bar(main_frame)

    def _create_menu(self):
        menubar = tk.Menu(self.root)
        self.root.config(menu=menubar)
        file_menu = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="File", menu=file_menu)
        file_menu.add_command(label="New Playlist", command=self.new_playlist, accelerator="Ctrl+N")
        file_menu.add_separator()
        file_menu.add_command(label="Open Files…", command=self.open_files, accelerator="Ctrl+O")
        file_menu.add_command(label="Open Playlist…", command=self.open_playlist, accelerator="Ctrl+Shift+O")
        file_menu.add_separator()
        file_menu.add_command(label="Save Playlist", command=self.save_playlist, accelerator="Ctrl+S")
        file_menu.add_command(label="Save Playlist As…", command=self.save_playlist_as, accelerator="Ctrl+Shift+S")
        file_menu.add_separator()
        file_menu.add_command(label="Exit", command=self.on_close, accelerator="Ctrl+Q")

    def _create_controls(self, parent):
        controls_frame = ttk.Frame(parent)
        controls_frame.grid(row=0, column=0, sticky="ew", pady=(0, 10))

        self.btn_prev = ttk.Button(controls_frame, text="⏮ Prev", width=8, command=self.on_prev)
        self.btn_play = ttk.Button(controls_frame, text="▶ Play", width=8, command=self.on_play_pause)
        self.btn_stop = ttk.Button(controls_frame, text="⏹ Stop", width=8, command=self.on_stop)
        self.btn_next = ttk.Button(controls_frame, text="⏭ Next", width=8, command=self.on_next)
        
        self.btn_prev.pack(side="left", padx=2)
        self.btn_play.pack(side="left", padx=2)
        self.btn_stop.pack(side="left", padx=2)
        self.btn_next.pack(side="left", padx=2)

        # --- MODIFIED --- Replaced checkbox with a shuffle button
        btn_shuffle = ttk.Button(controls_frame, text="🔀 Shuffle", command=self.on_shuffle_playlist)
        btn_shuffle.pack(side="left", padx=(20, 5))

        # Volume control on the right
        self.vol_var = tk.DoubleVar(value=80.0)
        scale_vol = ttk.Scale(controls_frame, from_=0, to=100, variable=self.vol_var, command=self.on_volume_change, length=150)
        scale_vol.pack(side="right", padx=5)
        ttk.Label(controls_frame, text="Volume").pack(side="right")

    def _create_progress_display(self, parent):
        progress_frame = ttk.Frame(parent)
        progress_frame.grid(row=1, column=0, sticky="ew", pady=5)
        progress_frame.columnconfigure(0, weight=1)
        
        self.progress_var = tk.DoubleVar()
        self.progress_bar = ttk.Scale(progress_frame, from_=0, to=100, variable=self.progress_var, orient="horizontal", command=self.on_seek)
        self.progress_bar.grid(row=0, column=0, sticky="ew")

        self.time_var = tk.StringVar(value="--:-- / --:--")
        lbl_time = ttk.Label(progress_frame, textvariable=self.time_var, width=15, anchor="e")
        lbl_time.grid(row=0, column=1, sticky="e", padx=(10, 0))

        self.progress_bar.bind("<ButtonPress-1>", lambda e: self.on_seek(e, start=True))
        self.progress_bar.bind("<ButtonRelease-1>", lambda e: self.on_seek(e, end=True))
    
    # ... (rest of UI creation methods are unchanged) ...
    def _create_playlist_view(self, parent):
        list_frame = ttk.Frame(parent)
        list_frame.grid(row=2, column=0, sticky="nsew", pady=5)
        list_frame.rowconfigure(1, weight=1)
        list_frame.columnconfigure(0, weight=1)
        drop_hint = "Drag & Drop Files/Folders Here" if DND_AVAILABLE else "Use File Menu to Open Files"
        lbl_hint = ttk.Label(list_frame, text=drop_hint, foreground="#888", padding=(5, 2))
        lbl_hint.grid(row=0, column=0, columnspan=2, sticky="w")
        self.listbox = tk.Listbox(list_frame, selectmode="browse", activestyle="dotbox", borderwidth=0, highlightthickness=0)
        self.listbox.grid(row=1, column=0, sticky="nsew")
        scrollbar = ttk.Scrollbar(list_frame, orient="vertical", command=self.listbox.yview)
        scrollbar.grid(row=1, column=1, sticky="ns")
        self.listbox.config(yscrollcommand=scrollbar.set)
        if DND_AVAILABLE:
            self.listbox.drop_target_register(DND_FILES)
            self.listbox.dnd_bind("<<Drop>>", self.on_drop)

    def _create_status_bar(self, parent):
        self.status_var = tk.StringVar(value="Ready")
        lbl_status = ttk.Label(parent, textvariable=self.status_var, anchor="w", padding=(5, 2), relief="sunken")
        lbl_status.grid(row=3, column=0, sticky="ew", pady=(10, 0))

    def _configure_bindings(self):
        self.root.bind_all("<Control-n>", lambda e: self.new_playlist())
        self.root.bind_all("<Control-o>", lambda e: self.open_files())
        self.root.bind_all("<Control-Shift-O>", lambda e: self.open_playlist())
        self.root.bind_all("<Control-s>", lambda e: self.save_playlist())
        self.root.bind_all("<Control-Shift-S>", lambda e: self.save_playlist_as())
        self.root.bind_all("<Control-q>", lambda e: self.on_close())
        self.root.bind("<space>", lambda e: self.on_play_pause())
        self.root.bind("<Right>", lambda e: self.on_next())
        self.root.bind("<Left>", lambda e: self.on_prev())
        self.listbox.bind("<Double-Button-1>", self.on_listbox_double_click)
        self.listbox.bind("<Return>", self.on_listbox_double_click)

    # --- File/Playlist Operations (mostly unchanged) ---
    def add_files_threaded(self, paths_to_scan):
        self.update_status("Scanning for audio files...")
        self.root.config(cursor="watch")
        thread = threading.Thread(target=self._scan_files_worker, args=(paths_to_scan,), daemon=True)
        thread.start()

    def _scan_files_worker(self, paths_to_scan):
        found_files = []
        for p_str in paths_to_scan:
            p = Path(p_str)
            if p.exists():
                found_files.extend(find_audio_files_in(p))
        self.root.after(0, self._process_scanned_files, found_files)

    def _process_scanned_files(self, found_files: list[Path]):
        added_count = self.playlist.add_paths(found_files)
        if added_count > 0:
            self.update_status(f"Added {added_count} new track(s).")
            self.refresh_listbox()
            self._autosave()
        else:
            self.update_status("No new audio files found.")
        self.root.config(cursor="")

    def on_drop(self, event):
        paths = parse_dnd_event_data(event.data)
        if paths:
            self.add_files_threaded(paths)

    def open_files(self):
        paths = filedialog.askopenfilenames(
            title="Select Audio Files",
            filetypes=[("Audio Files", "*.mp3 *.wav *.flac"), ("All files", "*.*")]
        )
        if paths:
            self.add_files_threaded(paths)

    def new_playlist(self):
        if self.playlist.tracks and not messagebox.askyesno("New Playlist", "Clear the current playlist?"):
            return
        self.stop_audio()
        self.playlist.clear()
        self.playlist_path = None
        self.refresh_listbox()
        self.update_status("Playlist cleared.")
        self._autosave()

    def open_playlist(self):
        path_str = filedialog.askopenfilename(
            title="Open Playlist",
            filetypes=[("Anthro Playlist", f"*{PLAYLIST_EXT}"), ("All files", "*.*")]
        )
        if not path_str:
            return
        
        path = Path(path_str)
        try:
            with path.open("r", encoding="utf-8") as f:
                data = json.load(f)
            self.stop_audio()
            self.playlist.from_dict(data)
            self.playlist_path = path
            self.refresh_listbox()
            # --- REMOVED --- setting shuffle_var
            self.vol_var.set(self.playlist.volume * 100.0)
            self.on_volume_change()
            self.update_status(f"Loaded playlist: {path.name} ({len(self.playlist.tracks)} tracks)")
        except (json.JSONDecodeError, KeyError, Exception) as e:
            messagebox.showerror("Open Failed", f"Could not open playlist file:\n{e}")

    def save_playlist_as(self):
        if not self.playlist.tracks:
            messagebox.showinfo("Nothing to Save", "The playlist is empty.")
            return False
        
        path_str = filedialog.asksaveasfilename(
            title="Save Playlist As",
            defaultextension=PLAYLIST_EXT,
            filetypes=[("Anthro Playlist", f"*{PLAYLIST_EXT}")],
            initialfile="playlist" + PLAYLIST_EXT
        )
        if not path_str:
            return False

        self.playlist_path = Path(path_str)
        return self.save_playlist()

    def save_playlist(self):
        if self.playlist_path is None:
            return self.save_playlist_as()
        
        try:
            data = self.playlist.to_dict()
            with self.playlist_path.open("w", encoding="utf-8") as f:
                json.dump(data, f, indent=2)
            self.update_status(f"Saved playlist to {self.playlist_path.name}")
            return True
        except Exception as e:
            messagebox.showerror("Save Failed", f"Could not save playlist:\n{e}")
            return False

    # --- Playback Control ---
    def play_index(self, index: int):
        if not PYGAME_AVAILABLE or not (0 <= index < len(self.playlist.tracks)):
            return
        
        path = self.playlist.tracks[index]
        try:
            pygame.mixer.music.load(str(path))
            self.current_track_duration = pygame.mixer.Sound(str(path)).get_length()
            pygame.mixer.music.play()
            
            self.playlist.current_index = index
            self.is_playing = True
            self.is_paused = False
            self.btn_play.config(text="⏸ Pause")
            self._highlight_current()
            self.update_now_playing_status()
            self._autosave()
        except Exception as e:
            messagebox.showerror("Playback Error", f"Could not play file:\n{path.name}\n\nError: {e}")
            self.stop_audio()
    
    # ... (other playback controls are mostly unchanged) ...
    def on_play_pause(self):
        if not PYGAME_AVAILABLE: return
        if not self.is_playing:
            idx_to_play = self.playlist.current_index
            if idx_to_play == -1 and self.playlist.tracks:
                idx_to_play = self.listbox.curselection()
                idx_to_play = idx_to_play[0] if idx_to_play else 0
            if idx_to_play != -1:
                self.play_index(idx_to_play)
        elif self.is_paused:
            pygame.mixer.music.unpause()
            self.is_paused = False
            self.btn_play.config(text="⏸ Pause")
            self.update_now_playing_status()
        else:
            pygame.mixer.music.pause()
            self.is_paused = True
            self.btn_play.config(text="▶ Resume")
            self.update_status("Paused")

    def on_stop(self):
        self.stop_audio()
        self.update_status("Stopped")

    def stop_audio(self):
        if PYGAME_AVAILABLE: pygame.mixer.music.stop()
        self.is_playing = False
        self.is_paused = False
        self.current_track_duration = 0
        self.btn_play.config(text="▶ Play")
        self.time_var.set("--:-- / --:--")
        self.progress_var.set(0)
        self.root.title(APP_NAME)
        self._highlight_current(clear_only=True)

    def on_next(self):
        if self.playlist.tracks:
            self.play_index(self.playlist.get_next_index())

    def on_prev(self):
        if self.playlist.tracks:
            self.play_index(self.playlist.get_prev_index())

    def on_volume_change(self, _=None):
        vol = max(0.0, min(1.0, self.vol_var.get() / 100.0))
        self.playlist.volume = vol
        if PYGAME_AVAILABLE: pygame.mixer.music.set_volume(vol)
    
    def on_seek(self, event, start=False, end=False):
        if start: self.is_seeking = True
        if self.is_seeking:
            if not self.is_playing or self.current_track_duration == 0:
                self.progress_var.set(0)
                if end: self.is_seeking = False
                return
            
            seek_pos_sec = self.progress_var.get() / 100 * self.current_track_duration
            if end:
                pygame.mixer.music.play(start=seek_pos_sec)
                if self.is_paused: pygame.mixer.music.pause()
                self.is_seeking = False

    # --- ADDED --- New handler for the shuffle button
    def on_shuffle_playlist(self):
        """Shuffles the playlist and refreshes the view."""
        if len(self.playlist.tracks) < 2:
            self.update_status("Not enough tracks to shuffle.")
            return
        
        self.playlist.shuffle_tracks()
        self.refresh_listbox()
        self.update_status("Playlist shuffled.")
        self._autosave()

    # --- UI Updates ---
    def refresh_listbox(self):
        self.listbox.delete(0, "end")
        for i, p in enumerate(self.playlist.tracks):
            self.listbox.insert("end", f"{i+1:02d}. {p.name}")
        self._highlight_current()

    def _highlight_current(self, clear_only=False):
        self.listbox.selection_clear(0, "end")
        idx = self.playlist.current_index
        if not clear_only and 0 <= idx < len(self.playlist.tracks):
            self.listbox.selection_set(idx)
            self.listbox.see(idx)
            self.listbox.activate(idx)

    def on_listbox_double_click(self, _=None):
        selection = self.listbox.curselection()
        if selection:
            self.play_index(selection[0])
    # ... (other UI update methods are unchanged) ...
    def update_status(self, text: str):
        self.status_var.set(text)

    def update_now_playing_status(self):
        idx = self.playlist.current_index
        if 0 <= idx < len(self.playlist.tracks):
            track_path = self.playlist.tracks[idx]
            status = f"Now Playing [{idx+1}/{len(self.playlist.tracks)}]: {track_path.name}"
            self.update_status(status)
            self.root.title(f"▶ {track_path.name} - {APP_NAME}")
        else:
            self.root.title(APP_NAME)
    
    def _update_playback_progress(self):
        if PYGAME_AVAILABLE and self.is_playing and not self.is_paused and not self.is_seeking:
            current_pos_ms = pygame.mixer.music.get_pos()
            if current_pos_ms != -1:
                current_pos_sec = current_pos_ms / 1000.0
                if self.current_track_duration > 0:
                    progress_percent = (current_pos_sec / self.current_track_duration) * 100
                    self.progress_var.set(progress_percent)
                    m, s = divmod(int(current_pos_sec), 60)
                    tm, ts = divmod(int(self.current_track_duration), 60)
                    self.time_var.set(f"{m:02d}:{s:02d} / {tm:02d}:{ts:02d}")
            if not pygame.mixer.music.get_busy():
                self.on_next()
        self.root.after(250, self._update_playback_progress)

    # --- Session Management ---
    def _autosave(self):
        ensure_config_dir()
        try:
            with AUTO_PATH.open("w", encoding="utf-8") as f:
                json.dump(self.playlist.to_dict(), f, indent=2)
        except Exception:
            pass

    def _load_session(self):
        if not AUTO_PATH.exists():
            return
        try:
            with AUTO_PATH.open("r", encoding="utf-8") as f:
                data = json.load(f)
            self.playlist.from_dict(data)
            self.refresh_listbox()
            # --- REMOVED --- loading shuffle state
            self.vol_var.set(self.playlist.volume * 100.0)
            self.on_volume_change()
            self.update_status(f"Restored last session ({len(self.playlist.tracks)} tracks).")
        except Exception:
            self.update_status("Could not restore last session.")

    def on_close(self):
        self._autosave()
        if PYGAME_AVAILABLE:
            pygame.quit()
        self.root.destroy()


def main():
    if DND_AVAILABLE:
        root = TkinterDnD.Tk()
    else:
        root = tk.Tk()
        print("Note: 'tkinterdnd2' not found. Drag-and-drop will be disabled.")
        print("Install with: pip install tkinterdnd2")

    app = PlayerApp(root)
    root.mainloop()


if __name__ == "__main__":
    main()